import puppeteerExtra from "puppeteer-extra";
import stealthPlugin from "puppeteer-extra-plugin-stealth";
import chromium from "@sparticuz/chromium";
import { PutObjectCommand, S3Client } from "@aws-sdk/client-s3";
import { Utilities } from "./code/common/utilities.js";
import { TradeLoginTestCaseHandler } from "./code/test-cases/trade/trade-login-test-case-handler.js";
import { UltraLoginTestCaseHandler } from "./code/test-cases/ultra/ultra-login-test-case-handler.js";
const client = new S3Client({});

async function executeTestCase(testCaseId) {
    puppeteerExtra.use(stealthPlugin);
    const browser = await puppeteerExtra.launch({
        args: chromium.args,
        defaultViewport: chromium.defaultViewport,
        executablePath: await chromium.executablePath(),
        headless: chromium.headless,
        ignoreHTTPSErrors: true
    });
    const page = await browser.newPage();

    let testCaseHandler;
    switch (testCaseId) {
        case 'trade_login':
        case 'trade_add_to_cart':
        case 'trade_footer_links':
        case 'trade_g4m_product':
        case 'trade_popular_product':
        case 'trade_product_categories':
        case 'trade_user_profile':
        case 'trade_promo_product':
        case 'trade_resources':
        case 'trade_top_links':
        case 'trade_qa_test':
            {
                testCaseHandler = new TradeLoginTestCaseHandler(page);
                break;
            }
        case 'ultra_dropship':
        case 'ultra_add_to_cart':
        case 'ultra_additional_options':
        case 'ultra_eddm_map':
        case 'ultra_eddm_custom_boxes':
            {
                testCaseHandler = new UltraLoginTestCaseHandler(page);
                break;
            }
        default: {
            testCaseHandler = new TradeLoginTestCaseHandler(page);
            break;
        }
    }
    await testCaseHandler.execute();

    let imageFile = await page.screenshot({
        timeout: 5000,
        clip: await testCaseHandler.pageBody.boundingBox()
    });
    let imageFileName = Utilities.generateFileName();
    const command = new PutObjectCommand({
        Bucket: "dev-qaal-bucket",
        Key: imageFileName,
        Body: imageFile,
        ACL: 'public-read'
    });
    const response = await client.send(command);
    console.log(`Response: ${response}`);
    const pages = await browser.pages();
    await Promise.all(pages.map(async (page) => page.close()));
    await browser.close();
    return `https://dev-qaal-bucket.s3.us-west-1.amazonaws.com/${imageFileName}`;
}

export const handler = async (event, context) => {
    const responseHeaders = {
        "Content-Type": "application/json",
        "Access-Control-Allow-Origin": "*",
        "Allow": "GET, OPTIONS, POST",
        "Access-Control-Allow-Methods": "GET, OPTIONS, POST",
        "Access-Control-Allow-Headers": "*"
    };
    try {
        const params = JSON.parse(event.body);
        const data = await executeTestCase(params.testCaseId);
        return {
            statusCode: 200,
            headers: responseHeaders,
            body: JSON.stringify(data)
        }
    } catch (error) {
        console.log("ERROR", error.message);
        return {
            statusCode: 200,
            headers: responseHeaders,
            body: JSON.stringify({
                error: error.message
            })
        }
    }
}
